/*** Eclipse Class Decompiler plugin, copyright (c) 2012 Chao Chen (cnfree2000@hotmail.com) ***/
package com.google.webp;

import java.lang.reflect.Field;
import java.nio.ByteBuffer;

import android.graphics.Bitmap;
import android.os.Build;
import android.util.Log;

/**
 * webp解码工具类，用于将webp解码成Bitmap
 * 
 * 建议采用WebPDecode方法解码，将根据图片类型自动生成最适合的图片格式，提高解码性能和内存使用。
 * 无阿尔法通道采用RGB565解码，存在阿尔法通道则采用ARGB888解码。RGB565解码过程中采用抖动处理，使体验达到最佳。
 * 
 * 提示：
 * Android系统原生显存为RGB565，任何图片（包括ARGB8888)在输出时都会成为RGB565输出，期间进行抖动处理，使肉眼很难区别。
 * 所以理论上抖动处理和android的抖动处理一致时，将无任何差别。
 * 差别体验主要在抖动处理算法上，如果抖动算法比android使用的SKia抖动算法更优，体验将好于android。
 * @author bokui
 *
 */
public class libwebp {
	static Field sBmpBufferF;
	
	static{
		if(Build.VERSION.SDK_INT >= Build.VERSION_CODES.ICE_CREAM_SANDWICH){
			try {
				sBmpBufferF = Bitmap.class.getDeclaredField("mBuffer");
			} catch (SecurityException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			} catch (NoSuchFieldException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			}
		}
	}
	/**
	 * 获取webp文件信息，该接口基本耗时为0
	 * @param paramArrayOfByte	webp裸数据
	 * @param paramLong			webp裸数据长度
	 * @param paramArrayOfInt1	webp宽接收数组
	 * @param paramArrayOfInt2	webp高接收数组
	 * @param paramArrayOfInt3	webp是否存在阿尔法通道接收数组
	 * @return
	 */
	public static int WebPGetFeatures(byte[] paramArrayOfByte, long paramLong,
			int[] paramArrayOfInt1, int[] paramArrayOfInt2, int[] paramArrayOfInt3) {
		return libwebpJNI.WebPGetFeatures(paramArrayOfByte, paramLong,
				paramArrayOfInt1, paramArrayOfInt2, paramArrayOfInt3);
	}
	
	/**
	 * 测试so文件是否加载成功，请在调用完loadLibrary后调用该方法，确认so文件已经正确load。失败，则无法使用该解码
	 * @return	true 成功 false 失败
	 */
	public static boolean testLibraryLoaded(){
		try
		{
			return libwebpJNI.testLibraryLoaded();
		}
		catch(Exception e)
		{
			return true;
		}
		catch(java.lang.UnsatisfiedLinkError e2)
		{
			return false;
		}
		catch(java.lang.Error e3)
		{
			return false;
		}
	}
	
	/**
	 * 该接口通过判断是否存在阿尔法通道，将webp图片解码成RGB565 Bitmap或ARGB8888 Bitmap
	 * 
	 * 该解码方法解码性能比系统库提高50%以上，而且存在抖动效果。无阿尔法通道建议采用该方法解码。可以节省一半内存使用。
	 * 注意：
	 * 		如果需要对图片进行一些特殊的处理，建议采用解成原始的ARGB8888,请调用WebPDecodeARGB方法。
	 * @param paramArrayOfByte	webp裸数据
	 * @param paramLong			webp裸数据长度
	 * @param paramArrayOfInt1	webp宽接收数组
	 * @param paramArrayOfInt2	webp高接收数组
	 * @param output			解码后图片内存地址
	 * @return
	 */
	public static Bitmap WebPDecode(byte[] paramArrayOfByte) {
		if(paramArrayOfByte == null)
			return null;
		
		int[] width = new int[] { 0 };
	    int[] height = new int[] { 0 };
	    int[] hasAlpha = new int[] { 0 };
	    libwebp.WebPGetFeatures(paramArrayOfByte, paramArrayOfByte.length, width, height,hasAlpha);

	    Bitmap bmp = null;
    	if(hasAlpha[0] == 0){
    		bmp = Bitmap.createBitmap(width[0], height[0], Bitmap.Config.RGB_565);
    		byte[] buffer = getBmpBuffer(bmp);
    		if(buffer == null){
    			buffer = new byte[width[0]*height[0]*2];
    			libwebpJNI.WebPDecodeRGB565Fast(paramArrayOfByte, paramArrayOfByte.length, width, height, buffer);
    			bmp.copyPixelsFromBuffer(ByteBuffer.wrap(buffer));
    		}else{
    			libwebpJNI.WebPDecodeRGB565Fast(paramArrayOfByte, paramArrayOfByte.length, width, height, buffer);
    		}
    	}else{
    		
    		bmp = Bitmap.createBitmap(width[0], height[0], Bitmap.Config.ARGB_8888);
    		byte[] buffer = getBmpBuffer(bmp);
    		if(buffer == null){
    			buffer = new byte[width[0]*height[0]*4];
    			libwebpJNI.WebPDecodeARGBFast(paramArrayOfByte, paramArrayOfByte.length, width, height, buffer);
    			bmp.copyPixelsFromBuffer(ByteBuffer.wrap(buffer));
    		}else{
    			libwebpJNI.WebPDecodeARGBFast(paramArrayOfByte, paramArrayOfByte.length, width, height, buffer);
    		}
    	}
    	return bmp;
	}
	
	/**
	 * 将图片解码成原始ARGB8888 Bitmap，比系统库提升50%性能
	 * 
	 * @param paramArrayOfByte	webp裸数据
	 * @param paramLong			webp裸数据长度
	 * @param paramArrayOfInt1	webp宽接收数组
	 * @param paramArrayOfInt2	webp高接收数组
	 * @param output			解码后图片内存地址
	 * @return
	 */
	public static Bitmap WebPDecodeARGB(byte[] paramArrayOfByte) {
		if(paramArrayOfByte == null)
			return null;
		Bitmap bmp = null;
		int[] width = new int[] { 0 };
	    int[] height = new int[] { 0 };
	    int[] hasAlpha = new int[] { 0 };
	    libwebp.WebPGetFeatures(paramArrayOfByte, paramArrayOfByte.length, width, height,hasAlpha);
    		
		bmp = Bitmap.createBitmap(width[0], height[0], Bitmap.Config.ARGB_8888);
		byte[] buffer = getBmpBuffer(bmp);
		if(buffer == null){
			buffer = new byte[width[0]*height[0]*4];
			libwebpJNI.WebPDecodeARGBFast(paramArrayOfByte, paramArrayOfByte.length, width, height, buffer);
			bmp.copyPixelsFromBuffer(ByteBuffer.wrap(buffer));
		}else{
			libwebpJNI.WebPDecodeARGBFast(paramArrayOfByte, paramArrayOfByte.length, width, height, buffer);
		}
		
    	return bmp;
	}
	
	private static byte[] getBmpBuffer(Bitmap bmp){
		try {
			if(sBmpBufferF == null)
				return null;
			return (byte[]) sBmpBufferF.get(bmp);
		} catch (Exception e) {
			Log.e("WebpBitmap", e.getMessage());
		}
		return null;
	}
}
